package dSelf;

import java.rmi.*;

/**
 * ServerInterface lists all accessible interfaces of the {@link Server}.
 */
public interface ServerInterface extends Remote {

 /**
  * Creates a new reference to the lobby of this dSelf-VM.
  * 
  * @return The ID of the new reference. 
  */ 
  public int connectToLobby() throws RemoteException;

 /**
  * The name of the object with this ID.
  *
  * @param ID The ID of the reference
  * @return The name of that object
  */ 
  public String getNameOf(int pos) throws RemoteException;

 /**
  * Returns the slots of the object with this ID.
  *
  * @param ID The ID of the reference
  * @return A slot vector with all slots.
  */ 
  public SerializedSlotVector getSlotsOf(int pos) throws RemoteException;

 /**
  * Returns the content of the slot with the given name of the object 
  * with the specified ID.
  *
  * @param ID The ID of the reference
  * @param slotName The name of the slot
  * @return The content of the demanded slot
  */ 
  public SerializedSO getSlotContentOf(int pos, String slotname) 
      throws RemoteException;
  
 /**
  * Removes the reference at the specified index.
  */   
  public void removeRemoteReference(int pos) throws RemoteException;
  
 /**
  * Removes all slots of the object with this ID.
  *
  * @param ID The ID of the reference
  */ 
  public void removeAllSlotsOf(int pos) throws RemoteException;

 /**
  * Removes the slot with the specified name of the object with 
  * this ID.
  * 
  * @param ID The ID of the reference
  * @param slotName The name of the slot 
  */  
  public void removeSlotOf(int pos, String slotName)
      throws RemoteException, dSelfException;

 /**
  * Adds some slots to the object with this ID. Slots with the same
  * name are replaced by the new ones.
  * 
  * @param ID The ID of the reference
  * @param VMName The name of the dSelf-VM, where the caller of this
  * method is located
  * @param remSlots The remote slots, that are added to the object with
  * the specified ID
  */ 
  public void addSlotsTo(int pos, String VMName, SerializedSlotVector slots)
      throws RemoteException, dSelfException;
	 
 /**
  * Adds some slots to the object with this ID. Slots with the same
  * name aren't replaced by the new ones.
  * 
  * @param ID The ID of the reference
  * @param VMName The name of the dSelf-VM, where the caller of this
  * method is located
  * @param remSlots The remote slots, that are added to the object with
  * the specified ID
  */ 
  public void addSlotsIfAbsentTo(int pos, String VMName, SerializedSlotVector slots)
      throws RemoteException, dSelfException;
	 
 /**
  * Returns the parent objects of the object with this ID.
  *
  * @param ID The ID of the reference
  * @return An arry with all parent objects
  */
  public SerializedSO[] getParentsOf(int pos) throws RemoteException;

 /**
  * Returns the length of the object vector with the specified ID.
  *
  * @param ID The ID of the reference
  */   
  public int getLengthOfObjectVector(int ID) throws RemoteException;

 /**
  * Replaces the content of the slot with the given slotname, located
  * in the object with this ID, with the new content.
  *
  * @param ID The ID of the reference
  * @param VMName The name of the dSelf-VM, where the caller of this
  * method is located
  * @param obj The new content of the slot
  */ 
  public void setContentOf(int ID, String VMName, SerializedSO obj) 
      throws RemoteException, dSelfException;
  
 /**
  * Returns the object vector at the specified position.
  *
  * @param ID The ID of the reference
  * @param pos The position in the object vector 
  * @return The object at this position
  */ 
  public SerializedSO getObjectOfObjectVector(int ID, int pos) 
      throws RemoteException, ArrayIndexOutOfBoundsException;

 /**
  * Puts an object at specified position in the object vector with this
  * ID.
  *
  * @param ID The ID of the reference
  * @param VMName The name of the dSelf-VM, where the caller of this
  * method is located
  * @param pos The position in the object vector 
  * @param so The object, that will be stored in the object vector   
  */ 
  public void putObjectOfObjectVector(int ID, String VMName, int pos, 
      SerializedSO so) throws RemoteException, dSelfException;
      
 /**
  * Locks the object at specified position in the object vector with this
  * ID.
  *
  * @param ID The ID of the reference
  * @param hash The hashcode of the remote object that will lock 
  */
  public void lockObject(int ID, int hash) throws RemoteException;
  
 /**
  * Unlocks the object at specified position in the object vector with this
  * ID.
  *
  * @param ID The ID of the reference
  * @param hash The hashcode of the remote object that will lock 
  */
  public void unlockObject(int ID, int hash) 
      throws RemoteException, dSelfException;
  
 /**
  * Returns the hashcode of the object with the given ID.
  *
  * @param ID The ID of the reference
  */    
  public int getHashCodeOf(int ID) throws RemoteException;    
  
 /**
  * Returns the name of the VM of the object with the given ID.
  *
  * @param ID The ID of the reference
  */    
  public String getLocationOf(int ID) throws RemoteException;    
}
