package dSelf;
 
import java.util.Vector;
import dSelfVM;

/**
 * The class ObjectVectorSO repesents the object vector of dSelf.
 * ObjectVectorSO is a primitive dSelf object and so it has only one
 * parent slot called "parent", that is shared by all object vectors.
 * The object, that "parent" refers to has initialy only one
 * parent slot, also called "parent" that refers to the lobby.
 * This object is dedicated to include the user-defined methods,
 * that are shared by all object vectors.
 */
public abstract class ObjectVectorSO extends PrimitiveSO implements RemoteReferenceSO{

 /** The shared parent object for all object vectors */
  protected static LocalOrdinarySO parent = 
    new LocalOrdinarySO(new ParentSlot("parent", dSelfVM.lobbySO));

 /**
  * The shared slots for all object vectors. It consists of only one slot
  * called "parent"
  */
  protected static SlotVector slotVector = 
    new SlotVector(new ParentSlot("parent",parent));   
    
 /**
  * The shared parent slots for all object vectors. It consists of only one slot
  * called "parent"
  */
  protected static Vector parVector = new Vector();   
  
  static{
    parVector.add(parent);   
  }
    
 /**
  * Returns a clone of this object vector.
  */
  public Object clone(){

    DataSO[] clonedVec = new DataSO[getLength()];

    for(int i=0; i<getLength(); i++)
      clonedVec[i] = getObjectAt(i);
        
    return new LocalObjectVectorSO(clonedVec);
  }

 /**
  * Returns the object at the specified position.
  *
  * @param pos The index of the demanded object
  * @return The object at the specified position
  */
  public abstract DataSO getObjectAt(int pos);

 /**
  * Puts an object at the specified position.
  *
  * @param pos The index, where this object is put
  * @param arg The object, that is put at the specified position
  */
  public abstract void putObjectAt(int pos, DataSO arg);

 /**
  * Returns the length of this object vector
  */
  public abstract int getLength();

 /**
  * Returns the slots of the object vectors. I.e, it's only one slot
  * called "parent".
  */   
  public SlotVector getSlotVector(){
    
    return slotVector; 
  }
  
 /**
  * Returns the shared parent slot for all dSelf object vectors.
  */  
  protected LocalOrdinarySO getParent(){
  
    return parent;
  }
    
 /**
  * Returns the parent slots of the object vectors. I.e, it's only one 
  * slot called "parent".
  */   
  public Vector getParentVector(){
  
    return parVector;
  }   

 /**
  * Returns the name of this object. 
  */
  public String getName(){
  
    return "objVector";
  }

 /**
  * The dispatcher for dSelf object vectors, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
      throws dSelfException, NonLocalReturnException{

    try{
     
      switch(msg.getMessageID()){

        case PrimMsg.AT:                     
	  return _At(msg.getFirstArg());

        case PrimMsg.ATPUT:                     
	  return _AtPut(msg.getFirstArg(), msg.getSecondArg());

        case PrimMsg.COPYRANGEDSTPOSSRCSRCPOSLENGTH:                     
	  return _CopyRangeDstPosSrcSrcPosLength(
	                msg.getFirstArg(), msg.getSecondArg(),
			msg.getThirdArg(), msg.getFourthArg());

        case PrimMsg.CLONEFILLER:                     
	  return _CloneFiller(msg.getFirstArg(), msg.getSecondArg());

        case PrimMsg.SIZE:                     
	  return _Size();
      }

      return super.dispatchPrimMsg(msg);       
    }catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Returns the value of the object vector at the specified position. If 
  * the position is to low or high, a badIndexError will return.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  * @return The object at the specified position
  */ 
  protected DataSO _At(dSelfObject arg) throws dSelfException{

    checkIfIntegerSO("_At:", arg);
    
    try{
      return getObjectAt(((IntegerSO)arg).value);
    }catch(ArrayIndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive \"_At:\""+
          " is out of bounds !", "At:", "badIndexError"); 
    }  
  }
  	
 /**
  * Puts at the specified position the given object. If the position
  * is to low or high, a 'badIndexError' will return. 
  *  	
  * @param at The argument type must be IntegerSO and is checked by
  * the method. It specifies the index in the vector
  * @param arg The object, that shall be put at the specified position
  * @return This object vector 
  */
  protected ObjectVectorSO _AtPut(dSelfObject at, dSelfObject arg) 
      throws dSelfException{
  
    checkIfIntegerSO("_At:Put: (1st argument)", at);

    try{
      putObjectAt(((IntegerSO)at).value, (DataSO)arg);
    }catch(ArrayIndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive \"_At:Put:\""+
          " is out of bounds !", "_At:Put:", "badIndexError"); 
    }  

    return this;
  }

 /**
  * Returns a clone (shallow copy) of this object vector, possibly
  * resized. The first argument sprecifies the length of the new 
  * object vector and the second arguments specifies the initial
  * value of extra elements if the result vector is longer than this
  * object vector.  Fails with 'badSizeError' if the first argument 
  * is negative.
  *
  * @param size The size for the new object vector. The argument 
  * type must be IntegerSO and is checked by the method
  * @param filler The possibly needed filler
  * @return Returns this object vector
  */
  protected ObjectVectorSO _CloneFiller(dSelfObject size, dSelfObject filler) 
      throws dSelfException{
  
    checkIfIntegerSO("_CloneFiller: (1st argument)", size);

    if(((IntegerSO)size).value < 0)
      throw new dSelfException("Value for size must be greater than 0 !",
          "_Clone:Filler:", "badSizeError");
      
    DataSO[] dVec =  new DataSO[((IntegerSO)size).value];
    
    for(int i=0; i<((IntegerSO)size).value; i++)
      dVec[i] = i<getLength() ? getObjectAt(i) : (DataSO)filler;
      
    return new LocalObjectVectorSO(dVec);  
  }

 /**
  * Copies the number of objects, specified by "length", into this object
  * vector at the position "dstPos" from the "srcVec" object vector at 
  * position "srcPos". May fail with "badIndexError".
  *
  * @param dstPos The argument type must be IntegerSO and is checked by
  * the method. It specifies the destination index in the vector
  * @param srcVec The argument type must be ObjectVectorSO and is checked
  * by the method
  * @param srcPos The argument type must be IntegerSO and is checked by
  * the method. It specifies the source index in the vector
  * @param length The argument type must be IntegerSO and is checked by
  * the method. 
  * @return Returns this object vector
  */   
  protected ObjectVectorSO _CopyRangeDstPosSrcSrcPosLength(
      dSelfObject dstPos, dSelfObject srcVec,
      dSelfObject srcPos, dSelfObject length) throws dSelfException{

    checkIfIntegerSO("_CopyRangeDstPos:Src:SrcPos:Length: (1st argument)", dstPos);
    checkIfObjectVectorSO("_CopyRangeDstPos:Src:SrcPos:Length: (2nd argument)", srcVec);
    checkIfIntegerSO("_CopyRangeDstPos:Src:SrcPos:Length: (3rd argument)", srcPos);
    checkIfIntegerSO("_CopyRangeDstPos:Src:SrcPos:Length: (4th argument)", length);

    try{ 
      for(int i=0 ; i<((IntegerSO)length).value; i++)
        putObjectAt(((IntegerSO)dstPos).value+i, 
	    ((ObjectVectorSO)srcVec).getObjectAt(((IntegerSO)srcPos).value+i)); 
    }catch(ArrayIndexOutOfBoundsException e){  
      throw new dSelfException("Argument of primitive "+
          "\"_CopyRangeDstPos:Src:SrcPos:Length:\" is out of bounds !",
	  "_CopyRangeDstPos:Src:SrcPos:Length", "badIndexError"); 
    }
    
    return this; 		 
  }		 

 /**
  * Prints this object vector at the front-end, that was chosen by the user.
  *
  * @return This object vector
  */
  protected IntegerSO _Size(){
  
    return new IntegerSO(getLength());
  }
}
