package dSelf;
 
import java.io.*;
import java.util.Vector;
import dSelfVM;
 
/**
 * The class FloatSO repesents the float numbers of dSelf.
 * FloatSO is a primitive dSelf object and so it has only one
 * parent slot called "parent", that is shared by all floats.
 * The object, that "parent" refers to has initialy only one
 * parent slot, also called "parent" that refers to the lobby.
 * This object is dedicated to include the user-defined methods,
 * (e.g. +, -, * ...) that are shared by all floats.
 */
public class FloatSO extends SerializedPrimitiveSO{
 
  /** The value of the float object */
  public float value = 0;

 /** The shared parent object for all floats */
  protected static LocalOrdinarySO parent = 
    new LocalOrdinarySO(new ParentSlot("parent", dSelfVM.lobbySO));

 /**
  * The shared slots for all floats. It consists of only one slot
  * called "parent"
  */
  protected static SlotVector slotVector = 
    new SlotVector(new ParentSlot("parent",parent));   
    
 /**
  * The shared parent slots for all floats. It consists of only one slot
  * called "parent"
  */
  protected static Vector parVector = new Vector();   
  
  static{
    parVector.add(parent);   
  }
  
 /**
  * Creates new dSelf float with the given Java float value
  */  
  public FloatSO(float floatValue){

    value = floatValue;
  }
    
 /**
  * Creates new dSelf float with the given dSelf float value
  */  
  public FloatSO(Float floatValue){

    value = floatValue.floatValue();
  }
    
 /**
  * Returns the slots of the floats. I.e, it's only one slot
  * called "parent".
  */   
  public SlotVector getSlotVector(){
    
    return slotVector; 
  }
  
 /**
  * Returns the shared parent slot for all dSelf floats.
  */  
  protected LocalOrdinarySO getParent(){
  
    return parent;
  }
    
 /**
  * Returns the parent slots of the floats. I.e, it's only one slot
  * called "parent".
  */   
  public Vector getParentVector(){
  
    return parVector;
  }   

 /**
  * Returns the value of this float in the form "float(...)".
  */
  public String getName(){
  
    return "float("+value+")";
  }

 /**
  * The dispatcher for dSelf floats, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
         throws dSelfException, NonLocalReturnException{

    try{
     
      switch(msg.getMessageID()){

        case PrimMsg.EQ:                     
          return _Eq(msg.getFirstArg());

        case PrimMsg.FLOATADD:                     
	  return _FloatAdd(msg.getFirstArg());
	
        case PrimMsg.FLOATASDOUBLE:                     
	  return _FloatAsDouble();
	
        case PrimMsg.FLOATASINT:                     
	  return _FloatAsInt();
	
        case PrimMsg.FLOATASLONG:                     
	  return _FloatAsLong();
	
        case PrimMsg.FLOATASSHORT:                     
	  return _FloatAsShort();

        case PrimMsg.FLOATCEIL:                     
	  return _FloatCeil();
	
        case PrimMsg.FLOATDIV:                     
	  return _FloatDiv(msg.getFirstArg());
	
        case PrimMsg.FLOATEQ:                     
	  return _FloatEQ(msg.getFirstArg());
	
        case PrimMsg.FLOATFLOOR:                     
	  return _FloatFloor();

        case PrimMsg.FLOATGE:                     
	  return _FloatGE(msg.getFirstArg());
	
        case PrimMsg.FLOATGT:                     
	  return _FloatGT(msg.getFirstArg());
	
        case PrimMsg.FLOATLE:                     
	  return _FloatLE(msg.getFirstArg());
	
        case PrimMsg.FLOATLT:                     
	  return _FloatLT(msg.getFirstArg());
	
        case PrimMsg.FLOATMOD:                     
	  return _FloatMod(msg.getFirstArg());
	
        case PrimMsg.FLOATMUL:                     
	  return _FloatMul(msg.getFirstArg());
	
        case PrimMsg.FLOATNE:                     
	  return _FloatNE(msg.getFirstArg());
	
        case PrimMsg.FLOATPRINTSTRINGPRECISION:                     
	  return _FloatPrintStringPrecision(msg.getFirstArg());

        case PrimMsg.FLOATPRINTSTRING:                     
	  return _FloatPrintString();
	
        case PrimMsg.FLOATROUND:                     
	  return _FloatRound();
	
        case PrimMsg.FLOATSUB:                     
	  return _FloatSub(msg.getFirstArg());
	
        case PrimMsg.FLOATTRUNCATE:                     
	  return _FloatTruncate();
      }

      return super.dispatchPrimMsg(msg);       

    }catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Checks, if the argument is equal to the value of this float object.
  * As distinct from Java 0.0 don't equals to -0.0 !
  *
  * @param arg The argument must be of type FloatSO
  */
  protected BooleanSO _Eq(dSelfObject arg) throws dSelfException{ 
   
    checkIfFloatSO("_Eq:", arg);
     
    // 0.0 == -0.0 is in Java true, but in SELF false. Thus it's
    // sign is checked by converting it to a string
    if(value == 0 && ((FloatSO)arg).value == 0)
      return BooleanSO.asBooleanSO(
        (new Float(value)).toString().charAt(0) == 
	(new Float(((FloatSO)arg).value)).toString().charAt(0)); 

    return BooleanSO.asBooleanSO(((FloatSO)arg).value == value);
  }  

 /**
  * Adds another dSelf float to this float
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected FloatSO _FloatAdd(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatAdd:", so);  
    return new FloatSO(value + ((FloatSO)so).value);  
  }  

 /**
  * Converts this dSelf float to a dSelf double.  
  */
  protected DoubleSO _FloatAsDouble(){
  
    return new DoubleSO((double) value); 
  }  

 /**
  * Converts this dSelf float to a dSelf integer.  
  */
  protected IntegerSO _FloatAsInt(){
  
    return new IntegerSO((int) value); 
  }  

 /**
  * Converts this dSelf float to a dSelf long.  
  */
  protected LongSO _FloatAsLong(){ 
  
    return new LongSO((long) value); 
  }  

 /**
  * Converts this dSelf float to a dSelf short.  
  */
  protected ShortSO _FloatAsShort(){ 
  
    return new ShortSO((short) value); 
  }  

 /**
  * The ceiling function for this float. Returns the greatest value
  * greater than or equal to the receiver by rounding towards positive
  * infinity.  
  */
  protected FloatSO _FloatCeil(){
  
    return new FloatSO((float) Math.ceil(value)); 
  }  

 /**
  * Divides this float through another float
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected FloatSO _FloatDiv(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatDiv:", so);

    if(((FloatSO)so).value == 0.0)  
      throw new dSelfException(" Division by zero -> "+value+" _FloatDiv: "+
                 ((FloatSO)so).value+" !", "_FloatDiv:", "divisionByZeroError");
  
    return new FloatSO(value / ((FloatSO)so).value);
  }  
  
 /**
  * Checks, if the argument is equal to the value of this float object.
  * As distinct from {@link #_Eq(dSelfObject)} 0.0 equals to -0.0 !
  *
  * @param arg The argument type must be _FloatEQ and is checked by
  * the method
  */
  protected BooleanSO _FloatEQ(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatEQ:", so);  
    return BooleanSO.asBooleanSO(value == ((FloatSO)so).value);
  }
  
 /**
  * The floor function for this float. Returns the greatest integral
  * value less than or equal to the receiver by rounding towards 
  * negative infinity.
  */
  protected FloatSO _FloatFloor(){
  
    return new FloatSO((float) Math.floor(value)); 
  }  

 /**
  * Checks, if this float is greater or equal than its argument.
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected BooleanSO _FloatGE(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatGE:", so);  
    return BooleanSO.asBooleanSO(value >= ((FloatSO)so).value);
  }
      
 /**
  * Checks, if this float is greater than its argument.
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected BooleanSO _FloatGT(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatGT:", so);  
    return BooleanSO.asBooleanSO(value > ((FloatSO)so).value);
  }
  
 /**
  * Checks, if this float is less or equal than its argument.
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected BooleanSO _FloatLE(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatLE:", so);  
    return BooleanSO.asBooleanSO(value <= ((FloatSO)so).value);
  }
  
 /**
  * Checks, if this float is less than its argument.
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected BooleanSO _FloatLT(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatLT:", so);  
    return BooleanSO.asBooleanSO(value < ((FloatSO)so).value);
  }

 /**
  * Returns this float modulo argument. 
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected FloatSO _FloatMod(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatMod:", so);  

    if(((FloatSO)so).value == 0.0)  
      throw new dSelfException(" Division by zero -> "+value+" _FloatMod: "
                       +((FloatSO)so).value+" !", "_FloatMod:", "divisionByZeroError");

    return new FloatSO(value % ((FloatSO)so).value); 
  }  
  
 /**
  * Returns the product of this float and the argument. 
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected FloatSO _FloatMul(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatMul:", so);  
    return new FloatSO(value * ((FloatSO)so).value); 
  }  
  
 /**
  * Checks if this float is inequal to its receiver
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */ 
  protected BooleanSO _FloatNE(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatNE:", so);  
    return BooleanSO.asBooleanSO(value != ((FloatSO)so).value);
  }

 /**
  * Returns the value of this float as a string.
  */ 
  protected StringSO _FloatPrintString(){
  
    return new StringSO(String.valueOf(value));
  }

 /** 
  * Analogous to {@link #_FloatPrintString()}, but takes an integer
  * argument, that specifies the number of digits after the decimal
  * point.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */ 
  protected StringSO _FloatPrintStringPrecision(dSelfObject so) throws dSelfException{
  
    checkIfIntegerSO("_FloatPrintStringPrecision:", so);
    
    return new StringSO(String.valueOf(
      ((float)((int)(value*Math.pow(10,((IntegerSO)so).value))))
           /Math.pow(10,((IntegerSO)so).value)));
  }

 /**
  * Returns this float rounded to the nearest integer. 0.5 is rounded
  * to even, so 1.5 rounds to 2, and 2.5 also rounds to 2.
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected IntegerSO _FloatRound(){
  
    return new IntegerSO(Math.round(value)); 
  }  

 /**
  * Returns this float minus the argument.
  *
  * @param arg The argument type must be FloatSO and is checked by
  * the method
  */
  protected FloatSO _FloatSub(dSelfObject so) throws dSelfException{

    checkIfFloatSO("_FloatSub:", so);  
    return new FloatSO(value - ((FloatSO)so).value); 
  }  

 /**
  * Returns this float towards zero. If it is greater than zero, then
  * the result is the same as {@link #_FloatFloor()}, otherwise it is
  * same as {@link #_FloatCeil()}.
  */
  protected FloatSO _FloatTruncate(){
  
    if(value > 0)
      return _FloatFloor();
    else 
      return _FloatCeil();   
  }  
}
