package dSelf;
 
import java.io.*;
import java.util.Vector;
import dSelfVM;

/**
 * The class DoubleSO repesents the double numbers of dSelf.
 * DoubleSO is a primitive dSelf object and so it has only one
 * parent slot called "parent", that is shared by all doubles.
 * The object, that "parent" refers to has initialy only one
 * parent slot, also called "parent" that refers to the lobby.
 * This object is dedicated to include the user-defined methods,
 * (e.g. +, -, * ...) that are shared by all doubles.
 */
public class DoubleSO extends SerializedPrimitiveSO{
 
  /** The value of the double object */
  public double value = 0;

 /** The shared parent object for all doubles */
  protected static LocalOrdinarySO parent = 
    new LocalOrdinarySO(new ParentSlot("parent", dSelfVM.lobbySO));

 /**
  * The shared slots for all doubles. It consists of only one slot
  * called "parent"
  */
  protected static SlotVector slotVector = 
    new SlotVector(new ParentSlot("parent",parent));   

 /**
  * The shared parent slots for all doubles. It consists of only one slot
  * called "parent"
  */
  protected static Vector parVector = new Vector();   
  
  static{
    parVector.add(parent);   
  }

 /**
  * Creates new dSelf double with the given Java double value
  */  
  public DoubleSO(double doubleValue){

    value = doubleValue;
  }
    
 /**
  * Creates new dSelf double with the given dSelf double value
  */  
  public DoubleSO(Double doubleValue){

    value = doubleValue.doubleValue();
  }
  
 /**
  * Returns the slots of the doubles. I.e, it's only one slot
  * called "parent".
  */   
  public SlotVector getSlotVector(){
    
    return slotVector; 
  }

 /**
  * Returns the shared parent slot for all dSelf doubles.
  */  
  protected LocalOrdinarySO getParent(){
  
    return parent;
  }
    
 /**
  * Returns the parent slots of the doubles. I.e, it's only one slot
  * called "parent".
  */   
  public Vector getParentVector(){
  
    return parVector;
  }   

 /**
  * Returns the value of this double in the form "double(...)".
  */
  public String getName(){
  
    return "double("+value+")";
  }

 /**
  * The dispatcher for dSelf doubles, that receives the primitive
  * messages, that are dedicated to it, and executes the actions.
  * 
  * @param msg The primitive message, that shall be executed
  * @return The result of the called primitive message
  */ 
  protected DataSO dispatchPrimMsg(PrimMsg msg) 
         throws dSelfException, NonLocalReturnException{

    try{
     
      switch(msg.getMessageID()){

        case PrimMsg.EQ:                     
	  return _Eq(msg.getFirstArg());

        case PrimMsg.DOUBLEADD:                     
	  return _DoubleAdd(msg.getFirstArg());
	
        case PrimMsg.DOUBLEASFLOAT:                     
	  return _DoubleAsFloat();
	
        case PrimMsg.DOUBLEASINT:                     
	  return _DoubleAsInt();
	
        case PrimMsg.DOUBLEASLONG:                     
	  return _DoubleAsLong();
	
        case PrimMsg.DOUBLEASSHORT:                     
	  return _DoubleAsShort();
	
        case PrimMsg.DOUBLECEIL:                     
	  return _DoubleCeil();
	
        case PrimMsg.DOUBLEDIV:                     
	  return _DoubleDiv(msg.getFirstArg());
	
        case PrimMsg.DOUBLEEQ:                     
	  return _DoubleEQ(msg.getFirstArg());
	
        case PrimMsg.DOUBLEFLOOR:                     
	  return _DoubleFloor();

        case PrimMsg.DOUBLEGE:                     
	  return _DoubleGE(msg.getFirstArg());
	
        case PrimMsg.DOUBLEGT:                     
	  return _DoubleGT(msg.getFirstArg());
	
        case PrimMsg.DOUBLELE:                     
	  return _DoubleLE(msg.getFirstArg());
	
        case PrimMsg.DOUBLELT:                     
	  return _DoubleLT(msg.getFirstArg());
	
        case PrimMsg.DOUBLEMOD:                     
	  return _DoubleMod(msg.getFirstArg());
	
        case PrimMsg.DOUBLEMUL:                     
	  return _DoubleMul(msg.getFirstArg());
	
        case PrimMsg.DOUBLENE:                     
	  return _DoubleNE(msg.getFirstArg());
	
        case PrimMsg.DOUBLEPRINTSTRINGPRECISION:                     
	  return _DoublePrintStringPrecision(msg.getFirstArg());

        case PrimMsg.DOUBLEPRINTSTRING:                     
	  return _DoublePrintString();
	
        case PrimMsg.DOUBLEROUND:                     
	  return _DoubleRound();
	
        case PrimMsg.DOUBLESUB:                     
	  return _DoubleSub(msg.getFirstArg());
	
        case PrimMsg.DOUBLETRUNCATE:                     
	  return _DoubleTruncate();
      }

      return super.dispatchPrimMsg(msg);       

    }catch(dSelfException e){
    
      return execFailBlock(msg, e);
    }
  }

 /**
  * Checks, if the argument is equal to the value of this double object.
  * As distinct from Java 0.0 don't equals to -0.0 !
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected BooleanSO _Eq(dSelfObject arg) throws dSelfException{ 

    checkIfDoubleSO("_Eq:", arg);   

    // 0.0 == -0.0 is in Java true, but in SELF false. Thus it's
    // sign is checked by converting it to a string
    if(value == 0 && ((DoubleSO)arg).value == 0)
      return BooleanSO.asBooleanSO(
        (new Double(value)).toString().charAt(0) == 
	(new Double(((DoubleSO)arg).value)).toString().charAt(0)); 

    return BooleanSO.asBooleanSO(((DoubleSO)arg).value == value);
  }  

 /**
  * Adds another dSelf double to this double
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected DoubleSO _DoubleAdd(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleAdd:", so);  
    return new DoubleSO(value + ((DoubleSO)so).value);  
  }  

 /**
  * Converts this dSelf double to a dSelf float.  
  */
  protected FloatSO _DoubleAsFloat(){
  
    return new FloatSO((float) value); 
  }  

 /**
  * Converts this dSelf double to a dSelf integer.  
  */
  protected IntegerSO _DoubleAsInt(){
  
    return new IntegerSO((int) value); 
  }  

 /**
  * Converts this dSelf double to a dSelf long.  
  */
  protected LongSO _DoubleAsLong(){
  
    return new LongSO((long) value); 
  }  

 /**
  * Converts this dSelf double to a dSelf short.  
  */
  protected ShortSO _DoubleAsShort(){
  
    return new ShortSO((short) value); 
  }  

 /**
  * The ceiling function for this double. Returns the greatest value
  * greater than or equal to the receiver by rounding towards positive
  * infinity.  
  */
  protected DoubleSO _DoubleCeil(){
  
    return new DoubleSO(Math.ceil(value)); 
  }  

 /**
  * Divides this double through another double
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected DoubleSO _DoubleDiv(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleDiv:", so);

    if(((DoubleSO)so).value == 0.0)  
      throw new dSelfException(" Division by zero -> "+value+" _DoubleDiv: "
                       +((DoubleSO)so).value+" !");
  
    return new DoubleSO(value / ((DoubleSO)so).value);
  }  
  
 /**
  * Checks, if the argument is equal to the value of this double object.
  * As distinct from {@link #_Eq(dSelfObject)} 0.0 equals to -0.0 !
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected BooleanSO _DoubleEQ(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleEQ:", so);  
    return BooleanSO.asBooleanSO(value == ((DoubleSO)so).value);
  }
  
 /**
  * The floor function for this double. Returns the greatest integral
  * value less than or equal to the receiver by rounding towards 
  * negative infinity.
  */
  protected DoubleSO _DoubleFloor(){
  
    return new DoubleSO(Math.floor(value)); 
  }  

 /**
  * Checks, if this double is greater or equal than its argument.
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected BooleanSO _DoubleGE(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleGE:", so);  
    return BooleanSO.asBooleanSO(value >= ((DoubleSO)so).value);
  }
      
 /**
  * Checks, if this double is greater than its argument.
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected BooleanSO _DoubleGT(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleGT:", so);  
    return BooleanSO.asBooleanSO(value > ((DoubleSO)so).value);
  }
  
 /**
  * Checks, if this double is less or equal than its argument.
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected BooleanSO _DoubleLE(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleLE:", so);  
    return BooleanSO.asBooleanSO(value <= ((DoubleSO)so).value);
  }
  
 /**
  * Checks, if this double is less than its argument.
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected BooleanSO _DoubleLT(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleLT:", so);  
    return BooleanSO.asBooleanSO(value < ((DoubleSO)so).value);
  }

 /**
  * Returns this double modulo argument. 
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected DoubleSO _DoubleMod(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleMod:", so);  
    if(((DoubleSO)so).value == 0.0)  
      throw new dSelfException(" Division by zero -> "+value+" _DoubleMod: "
                       +((DoubleSO)so).value+" !");

    return new DoubleSO(value % ((DoubleSO)so).value); 
  }  
  
 /**
  * Returns the product of this double and the argument. 
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected DoubleSO _DoubleMul(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleMul:", so);  
    return new DoubleSO(value * ((DoubleSO)so).value); 
  }  
 
 /**
  * Checks if this double is inequal to its receiver
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */ 
  protected BooleanSO _DoubleNE(dSelfObject so) throws dSelfException{

    checkIfDoubleSO("_DoubleNE:", so);  
    return BooleanSO.asBooleanSO(value != ((DoubleSO)so).value);
  }

 /**
  * Returns the value of this double as a string.
  */ 
  protected StringSO _DoublePrintString(){
  
    return new StringSO(String.valueOf(value));
  }

 /** 
  * Analogous to {@link #_DoublePrintString()}, but takes an integer
  * argument, that specifies the number of digits after the decimal
  * point.
  *
  * @param arg The argument type must be IntegerSO and is checked by
  * the method
  */ 
  protected StringSO _DoublePrintStringPrecision(dSelfObject so) throws dSelfException{
  
    checkIfIntegerSO("_DoublePrintStringPrecision:", so);
    return new StringSO(String.valueOf(
      ((double)((int)(value*Math.pow(10,((IntegerSO)so).value))))
         /Math.pow(10,((IntegerSO)so).value)));
  }

 /**
  * Returns this double rounded to the nearest integer. 0.5 is rounded
  * to even, so 1.5 rounds to 2, and 2.5 also rounds to 2.
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected LongSO _DoubleRound(){
  
    return new LongSO(Math.round(value)); 
  }  

 /**
  * Returns this double minus the argument.
  *
  * @param arg The argument type must be DoubleSO and is checked by
  * the method
  */
  protected DoubleSO _DoubleSub(dSelfObject so) throws dSelfException{
  
    return new DoubleSO(value - ((DoubleSO)so).value); 
  }  

 /**
  * Returns this double towards zero. If it is greater than zero, then
  * the result is the same as {@link #_DoubleFloor()}, otherwise it is
  * same as {@link #_DoubleCeil()}.
  */
  protected DoubleSO _DoubleTruncate(){
  
    if(value > 0)
      return _DoubleFloor();
    else 
      return _DoubleCeil();   
  }  
}
