package dSelf;

import java.util.Vector;
import dSelfVM;

/**
 * DataOrMethodSO declares the interfaces and implements the methods,
 * that are shared by data and method objects.
 */
public abstract class DataOrMethodSO extends dSelfObject{

 /** 
  * Returns the content of the slot with the specified name.
  *
  * @param slotName The name of the demanded slot
  * @return The content of the found slot or null if no one was found
  */
  public abstract dSelfObject getSlotContent(String slotName);

 /**
  * Returns the slot vector of this data object.
  */
  public abstract SlotVector getSlotVector();
  
 /**
  * Returns a vector with the contents of all parent slots.
  */
  public abstract Vector getParentVector();
  
 /**
  * Returns a clone of this object.
  */  
  public abstract Object clone();

 /**
  * The message dispatcher of this object. It receives the incoming
  * messages and forwards it to the specialized dispatcher for 
  * primitive ({@link #dispatchPrimMsg(PrimMsg)}) or ordinary 
  * ({@link #dispatchOrdinaryMsg(OrdinaryMsg)}) messages.
  *
  * @param msg The message, that is sent to this object
  * @return The result of the action, that was caused by the message
  */
  public DataSO dispatchMsg(Message msg) 
      throws dSelfException, NonLocalReturnException{

    if(msg instanceof OrdinaryMsg)
      return dispatchOrdinaryMsg((OrdinaryMsg) msg.setReceiver(this)); 

    return dispatchPrimMsg((PrimMsg) msg);
  }    

 /**
  * The message dispatcher for primitive messages. It checks, if the
  * message is understood by this object. If yes, then the actions
  * of this are done, otherwise an error message is thrown.
  *
  * @param msg The primitive message, that is sent to this object
  * @return The result of the action, that was caused by the message
  */ 
  protected abstract DataSO dispatchPrimMsg(PrimMsg msg) 
                  throws dSelfException, NonLocalReturnException;
  
 /**
  * The message dispatcher for ordinary messages. It checks, if the
  * message is understood by this object. If yes, then the content of
  * the slot, that matches with the selector of the given message, is
  * returned. Otherwise the message is delegated to its parents by
  * using the lookup algorithm.
  *
  * @param msg The ordinary message, that is sent to this object
  * @return The content of the found slot
  */
  protected DataSO dispatchOrdinaryMsg(OrdinaryMsg msg) 
               throws dSelfException, NonLocalReturnException{

    LookupResult lr = lookupMsg(msg);
    
//     if(Globals.debug_searchPath)
//       dSelfVM.printMessage("...evaluate slot \"" + msg.getSelector() + "\"");

    System.err.println("dispatchOrdinaryMsg(" + msg + ") should not occur any more");
    
    // Now evaluate the object 
    return evalSO(lr.getValue(), msg.getArgs(), msg.getSelector());      
  }

  protected LookupResult lookupMsg(OrdinaryMsg msg) 
               throws dSelfException{

    Vector foundObjs = delegate(msg);

    // If no object was found give an alert.
    if(foundObjs.size() == 0)
      throw new dSelfException("Message \""+msg.getSelector()+
                  "\" was not understood by object <"+getName()+":"+this+"> !");     
    
    // If more than 1 Slot was found, check if they are all nested 
    // in the same object. If they aren't, then the message is ambigious  
    if(foundObjs.size() > 1)
      for(int i=1; i<foundObjs.size(); i++)
        if(foundObjs.get(i) != foundObjs.get(0))
          throw new dSelfException("Message \""+msg.getSelector()+"\" send to object <"
                  +getName()+"> is ambigious !");        
    
    return (LookupResult)foundObjs.firstElement();
  }

 /**
  * Delegates a message to the parent objects of this object. The result
  * of this delegation is a vector with the contents of all found slots,
  * whoes names match with the selector of the message. If the vector is
  * empty, then no slot was found and if its size is bigger than 1 it's
  * ambigious, if the references inside don't refer all to the same object.
  *
  * @param msg The ordinary message, that is delegated.
  */ 
  public Vector delegate(OrdinaryMsg msg) throws dSelfException{

    // If the search path debugger is activated show some informations  
    if(Globals.debug_searchPath)
      dSelfVM.printMessage("...search \""+msg.getSelector()+"\" in <"+getName()+":"+this+">");

    // Enter this object as a node on our search path. If the message reached
    // this object twice an alert will be returned.
    if(!msg.addVisited(this, getLocation())){
      throw new dSelfException("Message cycling detected with \""+msg.getSelector()+
                       "\" at "+getName());
    }
    
    // Look for a matching slot
    Vector resultVec = new Vector();    
    dSelfObject so = getSlotContent(msg.getSelector());

    // If a matching slot was found, it is added to the result vector. 
    // Otherwise we keep on searching and send it to the parents.     
    if(so == null){
      // Send the message to all parents of this object
      Vector parSO = getParentVector();

      for(int par=0; par<parSO.size(); par++)
        resultVec.addAll(((DataOrMethodSO)parSO.get(par)).delegate(msg)); 
    }else{ 
      // A matching slot was found
      if(Globals.debug_searchPath)
        dSelfVM.printMessage("...found \"" + msg.getSelector() + "\" in <" +
	    getName() + ":" + this + ">");

      if(msg.isResend()){
        // The message was resended. So it will be forwarded as a non-resend message.

        if(msg.isDirectedResend()){
          // Look for the delegatee of directed resend 
          dSelfObject delegatee = getSlotContent(msg.getResendReceiver());
        
          if(delegatee == null){
            throw new dSelfException("Delegatee \"" + msg.getResendReceiver() + "\" " +
	        "of message \"" + msg.getSelector() + "\" was not found !");
  	  }else{
	    // Delegatee was found
	    if(delegatee instanceof MethodSO){
              throw new dSelfException("Delegatee \"" + msg.getResendReceiver() + "\" " +
	          "of message \"" + msg.getSelector() + "\" was not a data slot !");
            }else{   
 	      // Skip the found slot and continue searching for matching
              if(Globals.debug_searchPath)
                dSelfVM.printMessage("...directed resend \"" + msg.getSelector() + 
	            "\" to <" + ((DataOrMethodSO)delegatee).getName() + ":" + 
		    ((DataOrMethodSO)delegatee) + ">");
	      // slot at delegatee 
              msg.setNoMoreResend();
	      resultVec.addAll(((DataOrMethodSO)delegatee).delegate(msg));
	    }
	  }
        }else{
          // The message is an indirected resend.
	  // Skip this found slot and continue searching for matching
	  // slot at the parents
          if(Globals.debug_searchPath)
            dSelfVM.printMessage("...resend \"" + msg.getSelector() + 
	        "\" to all parents of <" + getName() + ": " + this + ">");

          // Send the message to all parents of this object
          Vector parSO = getParentVector();

          msg.setNoMoreResend();
          for(int par=0; par<parSO.size(); par++)
            resultVec.addAll(((DataOrMethodSO)parSO.get(par)).delegate(msg)); 
        }
      }else{
        // The message wasn't a resend
        resultVec.add(new LookupResult(this, so));
      }
    }
    
    // We make a backtrack now, so remove this object from the list
    if(Globals.debug_searchPath)
      dSelfVM.printMessage("...make backtrack in <"+getName()+":"+this+">");
      
    msg.removeVisited(this, getLocation());
      
    return resultVec;	    
  }
  
 /**
  * Evaluates the given dSelf object with the given arguments, that 
  * was found found in a slot with the given name and evaluates it
  * in the context of this object. 
  *
  * @param dSO The object, that shall be evaluated
  * @param args The arguments of the given object. 
  * @return The result of the evaluation of the given object
  */ 
  protected abstract DataSO evalSO(dSelfObject dSO, Vector args, String messagename) 
                   throws dSelfException, NonLocalReturnException;
 
}
